﻿using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.Scripting;
using UnityEngine.UI;
using TMPro;
using UnityEngine.EventSystems;

public class ObjectSpawner : MonoBehaviour
{
    public GameObject prefabToSpawn;
    [Range(0.001f, 0.05f)]
    public float spawnCooldown = 0.001f;
    [Range(1.0f, 10.0f)]
    public float objectLifetime = 10;
    [Range(50, 300)]
    public int start_pooled_obj = 250;
    public List<GameObject> object_pool = new List<GameObject>();

    public bool turn_on_obj_pool = false;

    bool spawning = false;

    public bool useWaitPool = false;
    private Dictionary<float, WaitForSeconds> waitDic;
    public TextMeshProUGUI textSpawnCooldown;
    public TextMeshProUGUI textObjectLifetime;
    public TextMeshProUGUI textStartPooledObj;
    public Toggle toggleObjectPooling;
    public Toggle toggleRun;
    public TextMeshProUGUI textGCMode;

    private void Awake()
    {
        GarbageCollector.GCModeChanged += (GarbageCollector.Mode mode) =>
        {
            textGCMode.text = mode.ToString();
        };
        
        
        #if !UNITY_EDITOR
        GarbageCollector.GCMode = GarbageCollector.Mode.Disabled;
        #endif

        waitDic = new Dictionary<float, WaitForSeconds>();
        InitializeUI();
    }

    private WaitForSeconds GetWait(float waitTime)
    {
        if (useWaitPool)
        {
            WaitForSeconds wait;
            if (waitDic.TryGetValue(waitTime, out wait))
            {
                return wait;
            }

            wait = new WaitForSeconds(waitTime);
            waitDic.Add(waitTime, wait);
            return wait;
        }
        else
        {
            return new WaitForSeconds(waitTime);
        }
    }

    void Update()
    {
        if (!spawning) {
            if(!turn_on_obj_pool)
            {
                StartCoroutine(WaitAndSpawn(spawnCooldown));
            }
            else
            {
                StartCoroutine(WaitAndSpawnFromPool(spawnCooldown));
            }
        }
    }

    IEnumerator WaitAndSpawn(float secondsToWaitFor)
    {
        spawning = true;
        // yield return new WaitForSeconds(secondsToWaitFor);
        yield return GetWait(secondsToWaitFor);
        GameObject spawned = Instantiate(prefabToSpawn, transform.position, Quaternion.identity);
        Destroy(spawned, objectLifetime);
        spawning = false;
    }

    IEnumerator WaitAndSpawnFromPool(float secondsToWaitFor)
    {
        spawning = true;
        // yield return new WaitForSeconds(secondsToWaitFor);
        yield return GetWait(secondsToWaitFor);
        
        // Instead of Instantiate()
        GameObject spawned = NextAvailableObjectInPool();

        // Position to spawn point before making active
        //   Reset Animation too, if any
        spawned.transform.position = transform.position;
        spawned.transform.rotation = Quaternion.identity;
        spawned.SetActive(true);

        // Instead of Destroy()
        StartCoroutine(ReturnToPool(spawned, objectLifetime));
        spawning = false;
    }

     GameObject NextAvailableObjectInPool()
    {
        for (int i = 0; i < object_pool.Count; i++)
        {
            if (! object_pool[i].activeInHierarchy)
            {
                return object_pool[i];
            }
        }
        object_pool[0].SetActive(false);
        return object_pool[0];
    }

    IEnumerator ReturnToPool(GameObject spawned, float objectLifetime)
    {
        // yield return new WaitForSeconds(objectLifetime);
        yield return GetWait(objectLifetime);
        spawned.SetActive(false);
    }

    private IEnumerator PreparePooling()
    {
        bool isEnabled = enabled;
        
        FindObjectOfType<EventSystem>().enabled = false;
        toggleRun.interactable = false;
        enabled = false;

        if (isEnabled)
            // yield return new WaitForSeconds(spawnCooldown + objectLifetime + 0.1f);
            yield return GetWait(spawnCooldown + objectLifetime + 0.1f);
        else
            yield return null;
   
        int poolCnt = object_pool.Count;
        if (start_pooled_obj < poolCnt)
        {
            for (int i = start_pooled_obj; i < poolCnt; i++)
            {
                GameObject.Destroy(object_pool[i]);
            }
            object_pool.RemoveRange(start_pooled_obj, poolCnt - start_pooled_obj);
        }
        else if (start_pooled_obj > poolCnt)
        {
            for (int i = 0; i < start_pooled_obj - poolCnt; i++)
            {
                GameObject spawned = Instantiate(prefabToSpawn, transform.position, Quaternion.identity);
                spawned.SetActive(false);
                object_pool.Add(spawned);
            }
        }
        
        enabled = isEnabled;
        toggleRun.interactable = true;
        FindObjectOfType<EventSystem>().enabled = true;
    }

    private void OnDestroy()
    {
        StopAllCoroutines();
    }

    #region  UI
    private void InitializeUI()
    {
        textSpawnCooldown.transform.GetComponentInChildren<Slider>().value = spawnCooldown;
        textObjectLifetime.transform.GetComponentInChildren<Slider>().value = objectLifetime;
        textStartPooledObj.transform.GetComponentInChildren<Slider>().value = (float)start_pooled_obj;

        toggleObjectPooling.isOn = turn_on_obj_pool;
        toggleRun.isOn = false;
    }

    public void OnSliderChanged_SpawnCooldown(float val)
    {
        textSpawnCooldown.transform.GetChild(0).GetComponent<TextMeshProUGUI>().text = string.Format("{0:0.000}", val);
        spawnCooldown = val;
    }

    public void OnSliderChanged_ObjectLifetime(float val)
    {
        textObjectLifetime.transform.GetChild(0).GetComponent<TextMeshProUGUI>().text = string.Format("{0:0.0}", val);
        objectLifetime = val;
    }
    
    public void OnSliderChanged_StartPooledObj(float val)
    {
        textStartPooledObj.transform.GetChild(0).GetComponent<TextMeshProUGUI>().text = string.Format("{0:0}", val);
        start_pooled_obj = (int)val;
    }

    public void OnToggled_ObjectPooling(bool isOn)
    {
        if (turn_on_obj_pool != isOn)
        {
            if (isOn)
                StartCoroutine(PreparePooling());

            turn_on_obj_pool = isOn;
        }
    }

    public void OnToggled_Run(bool isOn)
    {
        enabled = isOn;
        toggleRun.transform.GetChild(1).GetComponent<TextMeshProUGUI>().text = isOn ? "Running..." : "Stopped (Toggle to run)";
    }

    public void OnToggled_UseWaitPool(bool isOn)
    {
        this.useWaitPool = isOn;
    }
    #endregion
}
